/**
 * Chrome MV3 Popup Script Shim
 * Wraps browser.runtime.sendMessage to support callback-style calls
 * (Simpler than content-shim.js - no DOM marker needed in popup)
 */
console.log('[CarbonGuru Popup Shim] Loading...');

(function() {
    'use strict';

    if (typeof browser === 'undefined') {
        console.error('[CarbonGuru Popup Shim] ERROR: browser is undefined - polyfill not loaded?');
        return;
    }

    const originalSendMessage = browser.runtime.sendMessage.bind(browser.runtime);
    const MAX_RETRIES = 3;
    const RETRY_DELAY_MS = 500;
    const TIMEOUT_MS = 5000;

    /**
     * Send message with retry logic for MV3 service worker lifecycle
     */
    async function sendMessageWithRetry(message, retries = 0) {
        return new Promise((resolve, reject) => {
            const timeoutId = setTimeout(() => {
                reject(new Error('Message timeout - service worker may not be ready'));
            }, TIMEOUT_MS);

            originalSendMessage(message).then(
                (response) => {
                    clearTimeout(timeoutId);
                    resolve(response);
                },
                (error) => {
                    clearTimeout(timeoutId);
                    // Check if error is due to disconnected port (service worker not ready)
                    if (retries < MAX_RETRIES &&
                        (error.message?.includes('disconnected') ||
                         error.message?.includes('Receiving end does not exist'))) {
                        console.log(`[CarbonGuru Popup Shim] Retrying message (${retries + 1}/${MAX_RETRIES})...`);
                        setTimeout(() => {
                            sendMessageWithRetry(message, retries + 1).then(resolve, reject);
                        }, RETRY_DELAY_MS);
                    } else {
                        reject(error);
                    }
                }
            );
        });
    }

    browser.runtime.sendMessage = function(message, callbackOrOptions, callback) {
        // Detect callback-style: sendMessage(message, callback)
        if (typeof callbackOrOptions === 'function') {
            callback = callbackOrOptions;
            callbackOrOptions = undefined;
        }

        // If callback provided, convert to Promise-style with retry
        if (typeof callback === 'function') {
            sendMessageWithRetry(message).then(
                (response) => callback(response),
                (error) => {
                    console.warn('[CarbonGuru Popup Shim] sendMessage error:', error.message);
                    callback(undefined);
                }
            );
            return true; // Indicate async response (MV2 pattern)
        }

        // Otherwise, use Promise-style with retry
        return sendMessageWithRetry(message);
    };

    console.log('[CarbonGuru Popup Shim] Loaded successfully');

    /**
     * Keepalive utility for MV3 service worker
     *
     * Popup can use this during active operations to keep service worker alive.
     * Usage: if (window.cgKeepalive) window.cgKeepalive.start();
     */
    const cgKeepalive = {
        _port: null,
        _pingInterval: null,

        start() {
            if (this._port) return;
            try {
                this._port = browser.runtime.connect({ name: 'cg_keepalive' });
                this._port.onDisconnect.addListener(() => {
                    this._port = null;
                    this._clearPing();
                });
                this._pingInterval = setInterval(() => {
                    if (this._port) {
                        try { this._port.postMessage({ type: 'ping' }); } catch (e) { this._clearPing(); }
                    }
                }, 20000);
                console.log('[CarbonGuru Popup Shim] Keepalive started');
            } catch (e) {
                console.warn('[CarbonGuru Popup Shim] Keepalive failed:', e);
            }
        },

        stop() {
            this._clearPing();
            if (this._port) {
                try { this._port.disconnect(); } catch (e) {}
                this._port = null;
            }
            console.log('[CarbonGuru Popup Shim] Keepalive stopped');
        },

        _clearPing() {
            if (this._pingInterval) {
                clearInterval(this._pingInterval);
                this._pingInterval = null;
            }
        }
    };

    window.cgKeepalive = cgKeepalive;
})();
